/* PAIGE Import/Export Class Definitions. Copyright 1995-96 DataPak Software, Inc.
Software by Gar.  CODEWARRIER NOTE: This header file can only be used with .cpp programs
and/or .c programs with Activate C++ Compiler option set. */

#ifndef PGTXRCPP_H
#define PGTXRCPP_H

#include "Paige.h"
#include "pgEmbed.h"
#include "pgTxr.h"

/* The file import class for C++ers: */

class PaigeImportFilter {

	protected:
		pg_filetype			file_type;			// Type of file being imported
		file_ref			filemap;			// The file reference
		long				file_os;			// Source OS where file came from
		memory_ref			memory_file;		// Optional OS handle (if importing from "memory")
		pg_ref				import_pg;			// The pg_ref we are importing to
		paige_rec_ptr		import_pg_rec;		// The <used> pointer to paige record
		file_io_proc		io_proc;			// The I/O function for reading the file
		pg_globals_ptr		paige_globals;		// PAIGE and memmgr globals
		long				text_position;		// Position into pg_ref we are importing
		long				bytes_imported;		// Total bytes imported so far
		long				bytes_read;			// Total bytes read from file
		pg_error			io_result;			// Error result of last operation

	// Basic member functions:

	public:

		PaigeImportFilter ();
		virtual ~PaigeImportFilter ();

	// Initialization (common to all, non-overridable):
	
		pg_error pgInitImportFile (pg_globals_ptr globals,
							pg_file_unit fileref,
							memory_ref memory_unit,
							file_io_proc read_proc,
							long first_position,
							long last_position);
		
		pg_error pgImportFile (pg_ref pg,
							long pg_position,
							long import_flags,
							pg_boolean keep_selection,
							short draw_mode);

		long				import_bits;		// Data type(s) to import (can be set by app)
		long				max_buffer;			// Maximum buffer size, in bytes
		pg_char_ptr			font_cross_table;	// Table of cross-fonts
		pg_char_ptr			character_table;	// Table of cross-chars

	// Low-level I/O stuff:

	protected:
		long				feature_bits;	// What and how we are capable of importing
		pg_char_ptr			io_buffer;		// Low-level I/O buffer
		long				buffer_size;	// Size of buffer
		memory_ref			buffer_ref;		// Memory_ref that created buffer
		long				buffer_index;	// Index into buffer
		long				filepos;		// Current real file position
		long				file_begin;		// Beginning of file
		long				file_eof;		// End-of-file position
		pg_short_t			unicode_flag;	// Byte Order Mark if Unicode, otherwise zero

	// Translator information (can be mucked with):

	public:
		pg_translator		translator;			// File transfer information
		pg_char_ptr			output_ptr;			// Current pointer to read text buffer

	// Low-level member functions (cannot override):

		pg_error pgGetImportByte (pg_char_ptr the_byte);	// Return next file byte
		pg_error pgGetImportNBytes (long PG_FAR *num_bytes, pg_char_ptr bytes);
		pg_boolean OutputCharacter (pg_char the_byte);
		pg_boolean OutputString (pg_char_ptr the_str);
		pg_boolean FindFontInList (font_info_ptr font);

	// Overridable member functions (higher level):
		
		virtual pg_error pgVerifySignature (void);
		virtual pg_error pgPrepareImport (void);
		virtual pg_boolean pgReadNextBlock (void);
		virtual void PG_FAR * pgProcessEmbedData (memory_ref ref, long embed_type);
		virtual pg_error pgImportDone ();
		virtual void pgMapFont (font_info_ptr font, long importing_os, long current_os);
		virtual void pgMapChars (pg_char_ptr chars, long num_chars, long file_os, long current_os);
};

// Native import class (derived from PaigeImportFile):

class PaigeNativeImportFilter : public PaigeImportFilter {

	public:
		PaigeNativeImportFilter();
		virtual ~PaigeNativeImportFilter();

	// Overridable member functions (higher level):
	
	public:
		virtual pg_error pgPrepareImport (void);
		virtual pg_error pgVerifySignature (void);
		virtual pg_boolean pgReadNextBlock (void);
		virtual pg_error pgImportDone (void);
	
	private:
		pg_ref			original_pg;		// Original pg_ref if partial import
};


// RTF import class (derived from PaigeImportFile):

class PaigeRTFImportFilter : public PaigeImportFilter {

	public:
		PaigeRTFImportFilter();
		virtual ~PaigeRTFImportFilter();

	// Overridable member functions (higher level):
	
	public:
		virtual pg_error pgPrepareImport (void);
		virtual pg_error pgVerifySignature (void);
		virtual pg_boolean pgReadNextBlock (void);
		virtual pg_error pgImportDone ();
		virtual void ProcessInfoCommand (short command, short parameter);
		virtual void UnsupportedCommand (pg_char_ptr command, short parameter);

	private:
		memory_ref			style_stack;		// {bracketted state} style(s) list
		memory_ref			font_table;			// Table of fonts as given in RTF header
		memory_ref			color_table;		// Color table for text
		memory_ref			picture_ref;		// If non-NULL we are reading a picture
		style_info			current_style;		// The current text format setting(s)
		par_info			current_par;		// The current paragraph formats
		font_info			current_font;		// The current font
		pg_doc_info			doc_info;			// Paper, page info
		style_table_entry	stylesheet;			// The stylesheet we are processing
		style_table_ptr		current_stylesheet;	// Non-null if processing stylesheets
		short				stylesheet_index;	// Index to stylesheet name
		short				stylesheet_level;	// Stack-nested level for stylesheet proceess
		short				default_font;		// Default font number (from RTF file)
		short				def_font_set;		// Default font that has been set in the doc
		short				embed_width;		// Width of embedded item, if any
		short				embed_height;		// Height of embedded item, if any
		long				embed_binary;		// Bytecount of incoming binary (of BIN type)
		long				embed_type;			// Type of embedded item, if any
		metafile_struct		metafile;			// Metafile input params
		short				twips_width;		// Original width from rtf in TWIPS
		short				twips_height;		// Original height 
		short				twips_gwidth;		// Original picwgoal command width
		short				twips_gheight;		// Original pichgoal command

	// Local member functions to handle the import:

		long ReadCommand (short PG_FAR *param, pg_char_ptr terminator);
		pg_boolean PrepareRTFPicture (void);
		pg_boolean ReadRTFPicture (void);
		pg_boolean DoSpecialCharCommand (short command, short parameter);
		pg_boolean DoDestinationCommand (short command, short parameter);
		pg_boolean DoDocumentCommand (short command, short parameter);
		pg_boolean DoParagraphCommand (short command, short parameter);
		pg_boolean DoStyleCommand (short command, short parameter);
		void PushStyleStack (void);
		void PopStyleStack (void);
		void SkipCurrentLevel (void);
		void ProcessStylesheet (void);
};


/* Generic EXPORT (file write) class: */

class PaigeExportFilter {

	protected:
		pg_filetype			file_type;			// Type of file being exported
		long				file_version;		// Version of file (example - 0x00060000)
		file_ref			filemap;			// The file reference
		long				file_os;			// The target platform
		memory_ref			memory_file;		// Optional OS handle (if importing from "memory")
		pg_ref				export_pg;			// The pg_ref we are exporting from
		paige_rec_ptr		export_pg_rec;		// The <used> pointer to paige record
		file_io_proc		io_proc;			// The I/O function for reading the file
		pg_globals_ptr		paige_globals;		// PAIGE and memmgr globals
		select_pair			export_range;		// Text selection we are importing
		long				bytes_exported;		// Total bytes exported so far
		pg_error			io_result;			// Error result of last operation

	// Basic member functions:

	public:

		PaigeExportFilter ();
		virtual ~PaigeExportFilter ();

		pg_char				file_kind[KIND_STR_SIZE];	// Recommended filetype (set by constructor)

	// Initialization (common to all, non-overridable):
	
		pg_error pgInitExportFile (pg_globals_ptr globals,
							pg_file_unit fileref,
							memory_ref memory_unit,
							file_io_proc write_proc,
							long first_position);
		
		pg_error pgExportFile (pg_ref pg,
							select_pair_ptr range,
							long export_flags,
							pg_boolean selection_only);
		
		long				feature_bits;		// What and how we are capable of exporting
		long				export_bits;		// Data type(s) we are exporting
		long				max_buffer;			// Maximum holding buffer, in bytes

	// Low-level I/O stuff:

	protected:
		pg_char_ptr			io_buffer;		// Low-level I/O buffer
		long				buffer_size;	// Size of buffer
		memory_ref			buffer_ref;		// Memory_ref that created buffer
		memory_ref			unicode_ref;	// Memory_ref to convert from Unicode
		long				buffer_index;	// Index into buffer
		long				file_begin;		// Beginning of file
		long				filepos;		// Current real file position
		long				total_out;		// Total bytes out (written to file or not)
		pg_short_t			last_char_out;	// Last character type out

	// Translator information (can be mucked with):

	public:
		pg_translator		translator;			// File transfer information

	// Low-level functions, non-overridable
		
		void pgWriteByte (pg_char the_byte);	// Send byte to holding buffer
		void pgWriteNBytes (pg_char_ptr bytes, long num_bytes); // Send byte(s) to holding buffer
		void pgWriteDecimal (short decimal_value);
		void pgWriteHexByte (pg_char the_byte);
		void pgWriteString (pg_char_ptr the_str, pg_char prefix, pg_char suffix);
		void pgOutputBDC (short PG_FAR *value, short subtractor, pg_boolean PG_FAR *zero_suppress);
		void pgFlushOutputBuffer (void);  // Dump pending bytes in hold buffer

	// Overridable member functions (higher level)

		virtual pg_bits8_ptr pgPrepareEmbedData (embed_ref ref, long PG_FAR *byte_count,
										long PG_FAR *local_storage);
		virtual void pgReleaseEmbedData (embed_ref ref, long local_storage);
		virtual pg_error pgPrepareExport (void);
		virtual pg_boolean pgWriteNextBlock (void);
		virtual pg_error pgExportDone ();
};



// Native export class (derived from PaigeImportFile):

class PaigeNativeExportFilter : public PaigeExportFilter {

	public:
		PaigeNativeExportFilter();
		virtual ~PaigeNativeExportFilter();

	// Overridable member functions (higher level):
	
	public:
		virtual pg_error pgPrepareExport (void);
		virtual pg_boolean pgWriteNextBlock (void);
		virtual pg_error pgExportDone (void);
	
	private:
		pg_ref			copied_ref;
};



// RTF export class (derived from PaigeExportFilter):

class PaigeRTFExportFilter : public PaigeExportFilter {

	public:
		PaigeRTFExportFilter();
		virtual ~PaigeRTFExportFilter();

	// Overridable member functions (higher level):
	
	public:
		virtual pg_error pgPrepareExport (void);
		virtual pg_boolean pgWriteNextBlock (void);
		virtual pg_error pgExportDone ();
		virtual pg_error OutputHeaders ();
		virtual pg_error OutputFooters ();
		virtual pg_error OutputEmbed ();
		virtual pg_error OutputCustomParams();

		pg_char			def_stylename[FONT_SIZE + BOM_HEADER]; // Default "normal" stylesheet name

	private:
	// Local member functions to handle the import:

		memory_ref			style_stack;		// {bracketted state} style(s) list
		memory_ref			font_table;			// Font table (pre-built)
		memory_ref			color_table;		// Color table (pre-built)
		style_info			def_style;			// Default style_info
		par_info			def_par;			// Default par_info
		font_info			def_font;			// Default font_info
		long				count_last_cr;		// Byte counter since last CR
		long				def_state_level;	// Default style-state level

		void WriteCommand (pg_char_ptr rtf_table,
								short table_entry,
								short PG_FAR *parameter,
								pg_boolean delimeter);
		void OutputFontTable ();
		void OutputColorTable ();
		void OutputStylesheets ();
		void OutputNamedStyle (named_stylesheet_ptr style, short named_index);
		void OutputPageInfo ();
		void OutputStyleInfo (style_info_ptr style, style_info_ptr mask);
		void OutputParInfo (par_info_ptr par, par_info_ptr mask);
		void PushStyleStack (void);
		void PopStyleStack (void);
		pg_char_ptr GetFontType (font_info_ptr font);
		void OutputCR (pg_boolean unconditional);
		short PointConversion (short value, pg_boolean convert_resolution, pg_boolean x10);
};


typedef PaigeImportFilter PG_FAR *PaigeImportObject;
typedef PaigeExportFilter PG_FAR *PaigeExportObject;


#endif

